/**
 * Archive Community Page JavaScript
 */

document.addEventListener('DOMContentLoaded', function() {
    // Initialize all functionality
    initializeNavbarScroll();
    initializeBackToTop();
    initializeCommunitySearch();
    initializeFilterButtons();
});

/**
 * Navbar Scroll Effect
 */
function initializeNavbarScroll() {
    const navbar = document.querySelector('.navbar');
    if (!navbar) return;

    window.addEventListener('scroll', function() {
        if (window.scrollY > 100) {
            navbar.classList.add('scrolled');
        } else {
            navbar.classList.remove('scrolled');
        }
    });
}

/**
 * Back to Top Button
 */
function initializeBackToTop() {
    const backToTop = document.getElementById('backToTop');
    if (!backToTop) return;

    // Show/hide based on scroll position
    window.addEventListener('scroll', function() {
        if (window.scrollY > 300) {
            backToTop.classList.add('visible');
        } else {
            backToTop.classList.remove('visible');
        }
    });

    // Smooth scroll to top
    backToTop.addEventListener('click', function() {
        window.scrollTo({ top: 0, behavior: 'smooth' });
    });
}

/**
 * Community Search Enhancement
 */
function initializeCommunitySearch() {
    const searchInput = document.getElementById('communitySearch');
    const searchForm = searchInput ? searchInput.closest('form') : null;
    
    if (!searchInput || !searchForm) return;

    // Add debounced live search suggestion (optional enhancement)
    let searchTimeout;
    searchInput.addEventListener('input', function() {
        clearTimeout(searchTimeout);
        const query = this.value.trim();
        
        // Add visual feedback
        if (query.length > 2) {
            searchInput.classList.add('ring-2', 'ring-primary');
        } else {
            searchInput.classList.remove('ring-2', 'ring-primary');
        }
    });

    // Submit on Enter key
    searchInput.addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            e.preventDefault();
            searchForm.submit();
        }
    });
}

/**
 * Filter Button Enhancement
 */
function initializeFilterButtons() {
    const filterButtons = document.querySelectorAll('.filter-button');
    
    filterButtons.forEach(button => {
        // Add hover effect
        button.addEventListener('mouseenter', function() {
            if (!this.classList.contains('active')) {
                this.style.transform = 'translateY(-2px)';
            }
        });
        
        button.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    });
}

/**
 * Lazy Loading for Community Images (Progressive Enhancement)
 */
function initializeLazyLoading() {
    const images = document.querySelectorAll('.community-card img');
    
    if ('IntersectionObserver' in window) {
        const imageObserver = new IntersectionObserver((entries, observer) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const img = entry.target;
                    img.classList.add('fade-in');
                    observer.unobserve(img);
                }
            });
        });
        
        images.forEach(img => imageObserver.observe(img));
    }
}

/**
 * Community Card Hover Effects
 */
function initializeCommunityCards() {
    const cards = document.querySelectorAll('.community-card');
    
    cards.forEach(card => {
        // Add touch support for mobile
        card.addEventListener('touchstart', function() {
            this.classList.add('touch-hover');
        });
        
        card.addEventListener('touchend', function() {
            setTimeout(() => {
                this.classList.remove('touch-hover');
            }, 300);
        });
    });
}

/**
 * Handle AJAX Pagination (Optional Enhancement)
 */
function initializeAjaxPagination() {
    const paginationLinks = document.querySelectorAll('.page-numbers a');
    
    paginationLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            // Only for future AJAX implementation
            // For now, allow normal page navigation
        });
    });
}

/**
 * Utility function to get query parameters
 */
function getQueryParam(param) {
    const urlParams = new URLSearchParams(window.location.search);
    return urlParams.get(param);
}

/**
 * Utility function to update URL without page reload
 */
function updateURL(params) {
    const url = new URL(window.location);
    Object.keys(params).forEach(key => {
        if (params[key]) {
            url.searchParams.set(key, params[key]);
        } else {
            url.searchParams.delete(key);
        }
    });
    window.history.pushState({}, '', url);
}

/**
 * Initialize animations on scroll
 */
function initializeScrollAnimations() {
    const animatedElements = document.querySelectorAll('.community-card, .prose h3');
    
    if ('IntersectionObserver' in window) {
        const animationObserver = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    entry.target.classList.add('animated');
                }
            });
        }, {
            threshold: 0.1
        });
        
        animatedElements.forEach(el => animationObserver.observe(el));
    }
}

// Initialize additional features
document.addEventListener('DOMContentLoaded', function() {
    initializeLazyLoading();
    initializeCommunityCards();
    initializeAjaxPagination();
    initializeScrollAnimations();
});